/*
 
   Fishcamp DIO card driver for linux
 
   Copyright (C) 2001 Matt Clark matt.clark@nottingham.ac.uk
   Copyright (C) 2001-2002 fishcamp engineering www.fishcamp.com
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
 */

#include <linux/module.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/delay.h>

#include <linux/config.h>
#include <linux/pci.h>  

#include "dio.h"

/* define which parms can be altered */

static int dio_major=DIO_MAJOR;

MODULE_PARM(dio_major,"i");

/* function headers- usual to find them here in a driver*/
static int dio_ioctl(struct inode *, struct file *, unsigned int , unsigned long);
static ssize_t dio_read(struct file *, char *, size_t, loff_t *);
static ssize_t dio_write(struct file *, const char *, size_t, loff_t *);
static int dio_open(struct inode *, struct file *);
static int dio_release(struct inode *, struct file *);
inline static int dio_handle_buffer(dio_data_t * ,int);
inline static dio_delay(void);
static float dio_adconv(int);  

/* global flag for text or bin access */
static BOOL dio_text;

struct file_operations dio_fops = {
	read:		dio_read,
	write:		dio_write,
	ioctl:		dio_ioctl,
	open:		dio_open,
	release:	dio_release,
/* mmap method will have to wait- 
 * 	this would be useful for high performance I/O */
/*	mmap:		dio_mmap,	*/

	};

int init_module(void) {
int i,res;
volatile u32	temp;
struct pci_dev*	dev=NULL;
u32 cmdRegister;


/* refuse to load if PCI is not native to the machine */
#ifndef CONFIG_PCI
	printk("pcitest:pci bus is not supported on this computer\n");
	return -ENODEV;
#endif   
/* refuse to run if the pci libraries are not supported in this kernel */
/* This is deprecated under 2.4 */
	if(!pcibios_present()){
		printk(KERN_ERR "dio: pci bus is present BUT pci support is not enabled in kernel\n");
		return -ENODEV;
		}										
/* look for a DIO card */
	if(!(dev=pci_find_device(DIO_VENDOR,DIO_DEVICE,dev))){
		printk(KERN_ERR "dio: no dio pci card found in this machine\n");
		return -ENODEV;
		}
/* next error should <never> occur (here for coding erros only)*/
	if(dev==NULL){
		printk(KERN_ERR "dio: card found but dev==NULL\n");
		return -ENODEV;
		}
/* now we have the device and a pointer to its device structure
 * extract the numbers we need using the dev structure rather than
 * the configuration registers (change since 2.1.xx matt oct 2001)
 * in case the kernel has rewritten the locations 
 */
/* now I know that base_address[0] is config
 *		 base_address[1] is the FPGA fuse map and private device code
 *		 base_address[2] is the I/O ports (as pci memory)  
 */

/* kernel version sepcific code here */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
	/* while we here bomb out at older kernels */
	printk(KERN_ERR "dio:kernels earlier than 2.2.xx not supported \n");
	return -ENODEV;
#endif



/* On some platforms (Mac) the boot-up code or bios does not enable memory space
 * accesses to PCI cards by default.  We explicitly do this here in case it wasn't
 * done at boot time.  x86 boxes do not need this but it doesn't hurt.
 */

  /* get the card's COMMAND register */
  pci_read_config_dword(dev, PCI_COMMAND, &cmdRegister);

  /* set the MEMORY SPACE bit */
  cmdRegister |= PCI_COMMAND_MEMORY;
  printk("dio: init_module: setting card's COMMAND word - %08x\n",cmdRegister);

  /* write it back out to the card */
  pci_write_config_dword(dev, PCI_COMMAND, cmdRegister);




#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
	/* under kernel 2.2.xx base_address[] is a member of dev */
	/* use ioremap to get the memory */
#define GET_BASE	dio_base=(unsigned int *)ioremap(dev->base_address[2]&PCI_BASE_ADDRESS_MEM_MASK,DIO_IO_REGION_SIZE);
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	/* use  2.4 interface */
#define GET_BASE	/* should call pci_enable(dev...) here...*/ \
			 dio_base=ioremap(pci_resource_start(dev,2)&PCI_BASE_ADDRESS_MEM_MASK,DIO_IO_REGION_SIZE);
#endif

	GET_BASE
	if(dio_base==NULL){
		printk(KERN_ERR "dio: init: could not ioremap pci address space\n");
		return -ENODEV;
		}

	dio_cntl=(unsigned int *)((void *)dio_base+DIO_CNTL);
	PRINT_DEBUG(KERN_DEBUG "dio: init: DIO at %x and CNTL at %x\n",(int) dio_base,(int) dio_cntl);
	

/* configure global variables - note these should be in dio_open
 * and part of dio_data struct for multiple opens / cards
 * and should be protected for race conditions and reentrancy 
 * under multiple open /card / processor configs
 * should be SMP safe under single access locking matt 2001
 */
	dio_text=FALSE; 
#ifdef DIO_DELAY
	dio_delay_time=0;
#endif
/* now configure the control word */
	temp=readl(dio_cntl);
	PRINT_DEBUG(KERN_DEBUG "dio:init:cntl=%x\n",temp);
	writel(0,dio_cntl);
	dio_dir=DIO_ALL_READ&DIO_CNTL_PORT_MASK;
	temp=readl(dio_cntl);
	PRINT_DEBUG(KERN_DEBUG "dio:init:cntl=%x\n",temp);
	writel((dio_dir|dio_latch)&DIO_CNTL_PORT_MASK,dio_cntl);
	temp=readl(dio_cntl);
	PRINT_DEBUG(KERN_DEBUG "dio:init:cntl=%x\n",temp);

/* now register the char device - never sure what order is best */
	res=register_chrdev(dio_major,DIO_MOD_NAME,&dio_fops);
	if (res<0) return(res);
	dio_major=res;
/* now the driver is installed */
	printk("dio: init_module: Matt Clark Oct 2001- device %d\n",dio_major);
	return 0;
	}     
	
void cleanup_module(void){
int res;
volatile u32 temp;
	printk("dio: cleanup_module: Matt Clark Oct 2001- device %d\n",dio_major);
	/* release resources */
	if(dio_base!=NULL){
		iounmap(dio_base);
		dio_base=NULL;
		dio_cntl=NULL;
		}
	/* release char driver */
	res=unregister_chrdev(dio_major,DIO_MOD_NAME);
	}    

/* now the actual functions */
static ssize_t dio_read(struct file * file, char * buf, size_t count, loff_t *ppos){
int length,buff_size,i;
dio_data_t *data;

/* check to see if any reads are enabled */
	if(dio_n_read==0) return 0;
/* check to see if private data is available (should always be but caution is best) */
	if(!(file->private_data)) return 0;
	data=(dio_data_t *)file->private_data;
/* now functionality splits between text and binary */

	if(dio_text==TRUE){
		/* handle eof in text interface */
		if(data->eof_next==TRUE){
			data->eof_next=FALSE;
			return 0;
			}
		/* we are opened as text- truncate the count and return
		 * a text string- the rest is handled in open */
		length=dio_n_read*sizeof(u32);
		if(length>count) return 0;
		count=length;
		}

/* quantise count to be u32 size maximum / len is in 32 bit words here*/
/* quantise cout to sizeof(u32)*n ports */
	length=count/sizeof(u32);
	if((buff_size=dio_handle_buffer(data,length))<0) return buff_size;
	/* clip the length to buffer length units */
	if(buff_size<length)length=buff_size;
/*quantise the size according to the number of reads */
	length=dio_n_read*(int)(length/dio_n_read);

 	PRINT_DEBUG("About to read %d u32s with a delay of %dus\n",length,dio_delay_time);
	for(i=0;i<length;){
		if(DIO_PORT0&dio_ports_read)data->buffer[i++]=readl(dio_base+DIO_IO_PORT_0);
		if(DIO_PORT1&dio_ports_read)data->buffer[i++]=readl(dio_base+DIO_IO_PORT_1);
		if(DIO_PORT2&dio_ports_read)data->buffer[i++]=readl(dio_base+DIO_IO_PORT_2);
#ifdef DIO_DELAY
		dio_delay();
#endif
		}	
	PRINT_DEBUG(KERN_DEBUG "dio:read dio_dir=%x, dio_ports_read=%x\n",dio_dir,dio_ports_read);
	PRINT_DEBUG(KERN_DEBUG "dio:read: got %x\n",data->buffer[0]);

/* return stuff according to binary or text */
	if(dio_text==FALSE){
		length*=sizeof(u32);
		copy_to_user(buf,data->buffer,length);
		return length;
		}
	else{
		char text_buffer[DIO_TEXT_BUFF_SIZE],*pos;
		for(i=0,length=0;i<dio_n_read;i++){
			length+=sprintf(text_buffer+length,"%x ",data->buffer[i]);
			}
		text_buffer[length]='\n';
		text_buffer[++length]='\0';
		copy_to_user(buf,text_buffer,length);
		/* enable use of "cat"- first read returns data, second EOF, next data, next EOF etc*/
		/* make sure eof is issued next time */
		data->eof_next=TRUE;
		return length;
		}
	}

static ssize_t dio_write(struct file * file,const  char * buf, size_t count, loff_t *ppos){
int length,buff_size,err,i,trans;
dio_data_t *data;


/* check to see if any writes are enabled */
	if(dio_n_write==0) return -EINVAL;
/* check to see if private data is available (should always be but I don't want to oops)*/
	if(!(file->private_data)) return -EFAULT;
	data=(dio_data_t *)file->private_data;
	
	if(dio_text==TRUE){
		/* get a text buffer- fixed size -we check it can't overrun */
		char text_buffer[DIO_TEXT_BUFF_SIZE],*pos;
		
		/* as we are writing I have to interpret the input number as a string 
		 * not trivial as sscanf is not defined in the kernel */
		memset(text_buffer,'\0',DIO_TEXT_BUFF_SIZE);
		copy_from_user(text_buffer,buf,count>DIO_TEXT_BUFF_SIZE-1?DIO_TEXT_BUFF_SIZE-1:count);
		PRINT_DEBUG(KERN_DEBUG "dio:interping buffer %s\n",text_buffer);
		for(i=0,pos=text_buffer;i<dio_n_write;i++){
			/* strip white (and non hex) space manually*/
			while((!is_hex(*pos))&((*pos!='\0')&((*pos)!='\n')))pos++;
			if((*pos)=='\0')break;
			if((*pos)=='\n')break;
			PRINT_DEBUG(KERN_DEBUG "dio:write:text reading on %s\n",pos);
			data->buffer[i]=simple_strtoul(pos,&pos,16);
			PRINT_DEBUG(KERN_DEBUG "dio:write:text now on %s\n",pos);
			}	
		if(i!=dio_n_write){
			printk(KERN_ERR "dio:write:text: badly formed text input\n");
			return -EINVAL;
			}
		if(i>=DIO_TEXT_BUFF_SIZE-1){
			printk(KERN_ERR "dio:write:warning: overflowed text buffer\n");
			return -EINVAL;
			}
		PRINT_DEBUG(KERN_DEBUG "dio:write:text got %s",text_buffer);
		length=dio_n_write;
		/* lie to the functions- pretend we swallowed the whole buffer if it is bigger than DIO_TEXT_BUFF_SIZE */
		trans=count;
		}
	else{
/* in binary writes we assume a sequence port0, port1, port2, port 0 .....
 * determined by the flags dio_ports_write
 * we have to make sure that only data in quantum of dio_n_write are 
 * transferred to prevent losing sync between writes so if there are
 * extra words that don't fit into the quatum ignore them */
		/* quantise count to be u32 size maximum / len is in 32 bit words here*/
		/* quantise cout to sizeof(u32)*n ports */
		length=count/sizeof(u32);
		if((buff_size=dio_handle_buffer(data,length))<0) return buff_size;
		/* clip length if it is > buff_size*/
		if(length>buff_size)length=buff_size;
		/* quantise according to the number of port reads */
		length=dio_n_write*(int)(length/dio_n_write);	
		/* grab the data */
		copy_from_user(data->buffer,buf,length*sizeof(u32));
		PRINT_DEBUG(KERN_DEBUG "dio:bin:write: got %x\n",data->buffer[0]);
		PRINT_DEBUG(KERN_DEBUG "dio:bin:read dio_dir=%x, dio_ports_write=%x\n",dio_dir,dio_ports_write);
		trans=length*sizeof(u32);
		}
	/* write it to the ports */
	for(i=0;i<length;){
		if(DIO_PORT0&dio_ports_write)writel(data->buffer[i++],dio_base+DIO_IO_PORT_0);
		if(DIO_PORT1&dio_ports_write)writel(data->buffer[i++],dio_base+DIO_IO_PORT_1);
		if(DIO_PORT2&dio_ports_write)writel(data->buffer[i++],dio_base+DIO_IO_PORT_2);
#ifdef DIO_DELAY
		dio_delay();
#endif
		}	
	return trans;
	}



/* OPEN AND RELEASE */
/* we could use a static data structure instead of private data 
 * because we enforce global locking
 * however I prefer private data as it 
 * makes life easier in the lng run
 */
static int dio_open(struct inode *inode, struct file *file){ 
dio_data_t *data;
int err,minor;
u32	what=0;
	/* simply enforce single open access */
	/* global locking here prevents ioctl and hardreset...*/
	if(dio_locked==TRUE) return -EBUSY;
	dio_locked=TRUE;

	if((file->private_data=kmalloc(sizeof(dio_data_t),GFP_KERNEL))==NULL)
		return -ENOMEM;
	data=(dio_data_t *)file->private_data;
	/* get the initial buffer of 512 bytes */
	data->buffer=NULL;
	data->buff_size=0;
	if((err=dio_handle_buffer(data,DIO_BUFFER_SIZE))<0) return err;
	minor=MINOR(inode->i_rdev);

/* check how it has been opened using minor dev number */
	if(minor==MINOR_BIN){
		/* binary (usual)*/
		dio_text=FALSE;
		}
	else{
		/* Handy text interface for scripts- */
		/* here we are opened as text so use the minor number to
		 * get the port numbers and then set the read / write 
		 * permissions accordingly */
		/* do it only on a port wide basis- clunk I know but 
		 * convenient */
		dio_text=TRUE;
		if(minor>MINOR_TEXT){
			printk(KERN_ERR "dio:open:text:minor out of range\n");
			return -ENODEV;
			}
		if(minor&1)what|=DIO_PORT0;	
		if(minor&2)what|=DIO_PORT1;	
		if(minor&4)what|=DIO_PORT2;	
		PRINT_DEBUG(KERN_DEBUG "dio: opened as text\n");
		/* now use this to set dio_dir */
		if((file->f_flags & O_ACCMODE)==O_WRONLY){
			dio_dir|=what;
			dio_ports_write=what;
			dio_n_write=0;
			if(dio_ports_write&DIO_PORT0)dio_n_write++;
			if(dio_ports_write&DIO_PORT1)dio_n_write++;
			if(dio_ports_write&DIO_PORT2)dio_n_write++;   
			PRINT_DEBUG(KERN_DEBUG "dio:open:text:dio_dir=%x,dio_ports_write=%x,dio_n_write=%d\n",(int)dio_dir,(int)dio_ports_write,dio_n_write);
			}
		else{
			dio_dir&=((~what)&DIO_DIR_MASK);
			dio_ports_read=what;
			dio_n_read=0;
			if(dio_ports_read&DIO_PORT0)dio_n_read++;
			if(dio_ports_read&DIO_PORT1)dio_n_read++;
			if(dio_ports_read&DIO_PORT2)dio_n_read++;
			PRINT_DEBUG(KERN_DEBUG "dio:open:text:dio_dir=%x,dio_ports_read=%x,dio_n_read=%d\n",(int)dio_dir,(int)dio_ports_read,dio_n_read);
			}
		/* set the i/o port directions */
		writel((dio_dir|dio_latch)&DIO_CNTL_PORT_MASK,dio_cntl); 	
		/* set to read and then end!*/
		data->eof_next=FALSE;
		}
		
	MOD_INC_USE_COUNT;
	return 0;
	}

static int dio_release(struct inode *inode, struct file *file) { 
dio_data_t *data;
	MOD_DEC_USE_COUNT;	
/* release the device */
	dio_locked=FALSE;

	if(file->private_data){
		data=(dio_data_t *)file->private_data;
		if(data->buffer!=NULL){
			kfree(data->buffer);
			data->buffer=NULL;
			data->buff_size=0;
			}
		kfree(file->private_data);
		file->private_data=NULL;
		}
	return 0;
	}	

static int dio_ioctl(struct inode * inode,struct file *file, unsigned int cmd, unsigned long arg){
	switch(cmd){
		case DIO_HARDRESET:
			/* resets the usage count to 1
			   for use after a module crash
			   make sure nothing else has opened
			   the module */
			printk(KERN_INFO "dio: ioctl: HARDRESET: attempting mod_use=1 ");
			/* pull it to 0 */
			while(MOD_IN_USE)MOD_DEC_USE_COUNT;
			/* inc once for this open */
			MOD_INC_USE_COUNT;
			/* set locked to FALSE */
			dio_locked=FALSE;
			printk(KERN_INFO "dio: Done\n");
			break;
		case DIO_LATCH:
			dio_latch=DIO_CNTL_LATCH_MASK;
			writel((dio_dir|dio_latch)&DIO_CNTL_PORT_MASK,dio_cntl);
			break;
		case DIO_UNLATCH:
			dio_latch=0;
			writel((dio_dir|dio_latch)&DIO_CNTL_PORT_MASK,dio_cntl);
			break;
		case DIO_DIR:
			dio_dir=arg;
			writel((dio_dir|dio_latch)&DIO_CNTL_PORT_MASK,dio_cntl);
			break;
		case DIO_READ_SEQ:
			dio_ports_read=(int)arg;
			dio_n_read=0;
			if(dio_ports_read&DIO_PORT0)dio_n_read++;
			if(dio_ports_read&DIO_PORT1)dio_n_read++;
			if(dio_ports_read&DIO_PORT2)dio_n_read++;
			break;
		case DIO_WRITE_SEQ:
			dio_ports_write=arg;	
			dio_n_write=0;
			if(dio_ports_write&DIO_PORT0)dio_n_write++;
			if(dio_ports_write&DIO_PORT1)dio_n_write++;
			if(dio_ports_write&DIO_PORT2)dio_n_write++;
		case DIO_SET_DELAY:
#ifdef DIO_DELAY
			dio_delay_time=(int)arg;
#endif
			break;
		default:
			/* passively ignore other commands unless debugging */
			 PRINT_DEBUG(KERN_DEBUG "dio: ioctl: Unknown ioctl cmd %x\n",(int)cmd);
			}
	return 0;
	}  


/* implement that growing buffer / only use kernel memory is required*/
inline static int dio_handle_buffer(dio_data_t * data,int size){
	/* buffer is big enough */
	if(size<data->buff_size)return data->buff_size;
	
	/* buffer has reached its limit */
	if(data->buff_size==DIO_BUFF_LIMIT)return DIO_BUFF_LIMIT;
	
	/* asked for over the limit but the limit has not been reached yet */
	if(size>DIO_BUFF_LIMIT)size=DIO_BUFF_LIMIT;
	
	/* check to see if there was a buffer */
	if(data->buffer!=NULL) kfree(data->buffer);

	data->buffer=kmalloc(sizeof(u32)*size,GFP_KERNEL);
	if(data->buffer==NULL) return -ENOMEM;
	data->buff_size=size;
	PRINT_DEBUG(KERN_DEBUG "dio:handle_buffer:buffer is %d u32s\n",size);
	return size;
	}

inline static dio_delay(){
#ifdef DIO_DELAY
int jif;
	if(dio_delay_time){
		if(dio_delay_time<DIO_USECS_PER_HZ){
		/* busy wait for ~ the right time*/
			udelay(dio_delay_time);
			}
		else{
		/* sleep for ~ the right time */
			jif=(dio_delay_time*HZ)/1000000;
			if(jif<1)jif=1;
			current->state=TASK_INTERRUPTIBLE;
			schedule_timeout(jif);
			}
		}
#endif
	}
